package furny.swing.admin.tags;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;

import furny.entities.Tag;
import furny.entities.TagType;
import furny.furndb.FurnDBManager;
import furny.furndb.TagUpdateListener;

/**
 * Panel that displays a table of tags.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class TagPanel extends JPanel implements TagUpdateListener {
  private final TagComboBox tagTypeComboBox;
  private final JTable tagTable;
  private final TagTableModel model;

  private final List<Tag> tags = new ArrayList<Tag>();

  /**
   * Instantiates a new tag panel.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public TagPanel() {
    setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.NONE;
    constraints.weightx = 0.01d;
    constraints.weighty = 0.0d;
    constraints.gridwidth = 1;
    constraints.gridheight = 1;
    constraints.anchor = GridBagConstraints.EAST;

    final JLabel label = new JLabel("Type");
    add(label, constraints);

    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 0.0d;
    constraints.gridx++;

    tagTypeComboBox = new TagComboBox();
    tagTypeComboBox.addActionListener(new TagTypeChangeListener());

    add(tagTypeComboBox, constraints);

    constraints.gridx++;
    constraints.weightx = 1d;
    add(Box.createHorizontalGlue(), constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.gridwidth = 3;
    constraints.fill = GridBagConstraints.BOTH;

    model = new TagTableModel();

    tagTable = new JTable(model);
    tagTable.setDefaultRenderer(Tag.class, new TagCellRenderer());
    tagTable.setTableHeader(null);

    getTagTable()
        .setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
    getTagTable().setRowSelectionAllowed(true);

    add(new JScrollPane(tagTable), constraints);

    constraints.gridy++;
    constraints.weightx = 0.0d;
    constraints.weighty = 0.0d;
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.WEST;

    add(new JButton(new ActionAddTag()), constraints);

    constraints.gridy++;

    add(new JButton(new ActionRemoveTag()), constraints);

    FurnDBManager.getInstance().addTagUpdateListener(this);

    updateTags();
  }

  /**
   * Gets the tag table.
   * 
   * @return the tag table
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public JTable getTagTable() {
    return tagTable;
  }

  /**
   * Gets the selected tags.
   * 
   * @return the selected tags
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public List<Tag> getSelectedTags() {
    final List<Tag> list = new ArrayList<Tag>();

    final int[] rows = tagTable.getSelectedRows();
    for (final int i : rows) {
      list.add(tags.get(i));
    }

    return list;
  }

  @Override
  public void tagsUpdated(final TagType type, final List<Tag> newTags) {
    final TagType thisType = tagTypeComboBox.getSelectedTagType();

    if (thisType.equals(type)) {
      SwingUtilities.invokeLater(new Runnable() {
        @Override
        public void run() {
          tags.clear();
          tags.addAll(newTags);

          model.fireTableDataChanged();
        }
      });
    }
  }

  /**
   * Updates selected tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void updateTags() {
    final TagType type = tagTypeComboBox.getSelectedTagType();

    FurnDBManager.getInstance().updateTags(type);
  }

  /**
   * Listener to update tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class TagTypeChangeListener implements ActionListener {
    @Override
    public void actionPerformed(final ActionEvent e) {
      updateTags();
    }
  }

  /**
   * Table model for tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class TagTableModel extends DefaultTableModel {

    @Override
    public Object getValueAt(final int row, final int column) {
      return tags.get(row);
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex) {
      return Tag.class;
    }

    @Override
    public int getColumnCount() {
      return 1;
    }

    @Override
    public int getRowCount() {
      return tags.size();
    }
  }

  /**
   * Table cell renderer for tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class TagCellRenderer extends DefaultTableCellRenderer {
    @Override
    public Component getTableCellRendererComponent(final JTable table,
        final Object value, final boolean isSelected, final boolean hasFocus,
        final int row, final int column) {

      super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
          row, column);

      if (value instanceof Tag) {
        setText(((Tag) value).getName());
      } else {
        setText("");
      }

      return this;
    }
  }

  /**
   * Action to add a tag.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionAddTag extends AbstractAction {

    /**
     * Instantiates a new action to add a tag.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionAddTag() {
      super("New tag");
      putValue(SHORT_DESCRIPTION, "Create a new tag");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final String name = JOptionPane.showInputDialog(TagPanel.this,
          "Please type the name for the new tag", "Add tag",
          JOptionPane.QUESTION_MESSAGE);

      if (name != null && !name.isEmpty()) {
        for (final Tag t : tags) {
          if (name.equals(t.getName())) {
            JOptionPane.showMessageDialog(TagPanel.this,
                "This tag already exists", "Error", JOptionPane.ERROR_MESSAGE);
            return;
          }
        }

        final TagType type = (TagType) tagTypeComboBox.getSelectedItem();
        if (type != null) {
          final Tag tag = new Tag(type, name);
          FurnDBManager.getInstance().saveTag(tag);
          updateTags();
        }
      }
    }
  }

  /**
   * Action to remove a tag.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionRemoveTag extends AbstractAction {

    /**
     * Instantiates a new action to remove a tag.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionRemoveTag() {
      super("Delete Tags");
      putValue(SHORT_DESCRIPTION, "Delete tags");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final List<Tag> tags = getSelectedTags();
      if (!tags.isEmpty()) {
        final int n = JOptionPane
            .showConfirmDialog(
                TagPanel.this,
                "Are you sure to delete the selected tags?\nThis can not be undone.",
                "Delete tag", JOptionPane.YES_NO_OPTION,
                JOptionPane.WARNING_MESSAGE);

        if (n == JOptionPane.YES_OPTION) {
          for (final Tag t : tags) {
            FurnDBManager.getInstance().deleteTag(t);
          }
          updateTags();
        }
      }
    }
  }
}
